const express = require('express');
const router = express.Router();
const emailService = require('../utils/emailService');
const { validateEnquiryForm, handleValidationErrors } = require('../middleware/validation');

// Enquiry form submission
router.post('/', validateEnquiryForm, handleValidationErrors, async (req, res) => {
  try {
    const { name, email, phone, enquiry, category, title } = req.body;

    // Log the submission (optional)
    console.log('Enquiry form submission received:', {
      name,
      email,
      phone: phone.replace(/\d(?=\d{4})/g, '*'), // Mask phone number in logs
      category,
      title,
      timestamp: new Date().toISOString()
    });

    // Send email
    const emailResult = await emailService.sendEnquiryEmail({
      name,
      email,
      phone,
      enquiry,
      category,
      title
    });

    console.log('Enquiry email sent successfully:', emailResult.messageId);

    res.status(200).json({
      success: true,
      message: 'Enquiry submitted successfully. We will review and respond soon!',
      data: {
        submissionId: emailResult.messageId,
        category,
        title,
        timestamp: new Date().toISOString()
      }
    });

  } catch (error) {
    console.error('Error processing enquiry form:', error);
    
    res.status(500).json({
      success: false,
      message: 'Failed to submit enquiry. Please try again later.',
      error: process.env.NODE_ENV === 'development' ? error.message : undefined
    });
  }
});

// Get enquiry categories (optional endpoint for frontend)
router.get('/categories', (req, res) => {
  const categories = [
    'General Inquiry',
    'Technical Support',
    'Sales',
    'Partnership',
    'Feedback',
    'Complaint',
    'Feature Request',
    'Other'
  ];

  res.status(200).json({
    success: true,
    data: categories
  });
});

// Health check for enquiry route
router.get('/health', (req, res) => {
  res.status(200).json({
    success: true,
    message: 'Enquiry service is running',
    timestamp: new Date().toISOString()
  });
});

module.exports = router;
