const { body, validationResult } = require('express-validator');

// Common validation rules
const nameValidation = body('name')
  .trim()
  .isLength({ min: 2, max: 50 })
  .withMessage('Name must be between 2 and 50 characters')
  .matches(/^[a-zA-Z\s]+$/)
  .withMessage('Name can only contain letters and spaces');

const emailValidation = body('email')
  .trim()
  .isEmail()
  .normalizeEmail()
  .withMessage('Please provide a valid email address');

const phoneValidation = body('phone')
  .trim()
  .isMobilePhone('any', { strictMode: false })
  .withMessage('Please provide a valid phone number');

const enquiryValidation = body('enquiry')
  .trim()
  .isLength({ min: 10, max: 1000 })
  .withMessage('Enquiry must be between 10 and 1000 characters');

// Contact form validation
const validateContactForm = [
  nameValidation,
  emailValidation,
  phoneValidation,
  enquiryValidation
];

// Enquiry form validation
const validateEnquiryForm = [
  nameValidation,
  emailValidation,
  phoneValidation,
  enquiryValidation,
  body('category')
    .trim()
    .isLength({ min: 2, max: 50 })
    .withMessage('Category must be between 2 and 50 characters'),
  body('title')
    .trim()
    .isLength({ min: 5, max: 100 })
    .withMessage('Title must be between 5 and 100 characters')
];

// Middleware to handle validation errors
const handleValidationErrors = (req, res, next) => {
  const errors = validationResult(req);
  
  if (!errors.isEmpty()) {
    return res.status(400).json({
      success: false,
      message: 'Validation failed',
      errors: errors.array().map(error => ({
        field: error.path,
        message: error.msg,
        value: error.value
      }))
    });
  }
  
  next();
};

module.exports = {
  validateContactForm,
  validateEnquiryForm,
  handleValidationErrors
};
