# VEN Backend API

A Node.js backend service for handling contact us and enquiry forms with Gmail integration. This backend is designed to be hosted on cPanel and includes CORS support.

## Features

- **Contact Form API**: Handle contact form submissions with name, email, phone, and enquiry fields
- **Enquiry Form API**: Handle enquiry submissions with additional category and title fields
- **Gmail Integration**: Automatically send formatted emails via Gmail
- **Input Validation**: Comprehensive validation for all form fields
- **CORS Support**: Configurable CORS for frontend integration
- **Rate Limiting**: Protection against spam and abuse
- **Security**: Helmet.js for security headers
- **Error Handling**: Comprehensive error handling and logging
- **cPanel Ready**: Optimized for cPanel hosting

## API Endpoints

### Contact Form
- **POST** `/api/contact`
  - Submit a contact form
  - Fields: `name`, `email`, `phone`, `enquiry`

### Enquiry Form
- **POST** `/api/enquiry`
  - Submit an enquiry form
  - Fields: `name`, `email`, `phone`, `enquiry`, `category`, `title`

### Utility Endpoints
- **GET** `/` - Health check
- **GET** `/api/contact/health` - Contact service health check
- **GET** `/api/enquiry/health` - Enquiry service health check
- **GET** `/api/enquiry/categories` - Get available enquiry categories

## Installation

### Prerequisites
- Node.js (v14 or higher)
- Gmail account with App Password enabled

### Local Development

1. **Clone and setup:**
   ```bash
   cd VEN-backend
   npm install
   ```

2. **Configure environment variables:**
   ```bash
   cp .env.example .env
   ```
   
   Edit `.env` file with your configuration:
   ```
   PORT=3000
   NODE_ENV=development
   GMAIL_USER=your-gmail@gmail.com
   GMAIL_APP_PASSWORD=your-16-character-app-password
   RECIPIENT_EMAIL=your-email@gmail.com
   ALLOWED_ORIGINS=http://localhost:3000,http://localhost:3001
   ```

3. **Start the server:**
   ```bash
   npm run dev    # Development with nodemon
   npm start      # Production
   ```

### Gmail App Password Setup

1. Enable 2-factor authentication on your Gmail account
2. Go to Google Account settings → Security → App passwords
3. Generate a new app password for "Mail"
4. Use this 16-character password in `GMAIL_APP_PASSWORD`

## cPanel Deployment

### File Upload
1. Upload all files to your cPanel public_html directory (or subdirectory)
2. Ensure Node.js is enabled in cPanel
3. Install dependencies via cPanel terminal or file manager

### Environment Configuration
1. Create `.env` file in your cPanel directory
2. Set production values:
   ```
   PORT=3000
   NODE_ENV=production
   GMAIL_USER=your-gmail@gmail.com
   GMAIL_APP_PASSWORD=your-app-password
   RECIPIENT_EMAIL=your-email@gmail.com
   ALLOWED_ORIGINS=https://yourdomain.com,https://www.yourdomain.com
   ```

### cPanel Setup
1. In cPanel, go to "Node.js Selector"
2. Create a new Node.js app:
   - Node.js version: 14+ (recommended: Latest LTS)
   - Application root: Your upload directory
   - Application URL: Your desired URL
   - Application startup file: `server.js`
3. Install dependencies by running `npm install` in the terminal
4. Start the application

## API Usage Examples

### Contact Form Submission
```javascript
// Frontend JavaScript example
const submitContact = async (formData) => {
  try {
    const response = await fetch('https://yourapi.com/api/contact', {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
      },
      body: JSON.stringify({
        name: 'John Doe',
        email: 'john@example.com',
        phone: '+1234567890',
        enquiry: 'I would like to know more about your services.'
      })
    });
    
    const result = await response.json();
    console.log(result);
  } catch (error) {
    console.error('Error:', error);
  }
};
```

### Enquiry Form Submission
```javascript
const submitEnquiry = async (formData) => {
  try {
    const response = await fetch('https://yourapi.com/api/enquiry', {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
      },
      body: JSON.stringify({
        name: 'Jane Smith',
        email: 'jane@example.com',
        phone: '+1234567890',
        enquiry: 'I need technical support for my account.',
        category: 'Technical Support',
        title: 'Account Access Issue'
      })
    });
    
    const result = await response.json();
    console.log(result);
  } catch (error) {
    console.error('Error:', error);
  }
};
```

## Response Format

### Success Response
```json
{
  "success": true,
  "message": "Form submitted successfully",
  "data": {
    "submissionId": "message-id",
    "timestamp": "2025-06-01T12:00:00.000Z"
  }
}
```

### Error Response
```json
{
  "success": false,
  "message": "Validation failed",
  "errors": [
    {
      "field": "email",
      "message": "Please provide a valid email address",
      "value": "invalid-email"
    }
  ]
}
```

## Validation Rules

### Contact Form
- **Name**: 2-50 characters, letters and spaces only
- **Email**: Valid email format
- **Phone**: Valid phone number format
- **Enquiry**: 10-1000 characters

### Enquiry Form
- All contact form validations plus:
- **Category**: 2-50 characters
- **Title**: 5-100 characters

## Security Features

- **Rate Limiting**: 100 requests per 15 minutes per IP
- **Input Validation**: Comprehensive validation using express-validator
- **Security Headers**: Helmet.js for security headers
- **CORS Protection**: Configurable allowed origins
- **Data Sanitization**: Email normalization and input trimming

## Environment Variables

| Variable | Description | Required | Default |
|----------|-------------|----------|---------|
| `PORT` | Server port | No | 3000 |
| `NODE_ENV` | Environment mode | No | development |
| `GMAIL_USER` | Gmail username | Yes | - |
| `GMAIL_APP_PASSWORD` | Gmail app password | Yes | - |
| `RECIPIENT_EMAIL` | Email recipient | No | GMAIL_USER |
| `ALLOWED_ORIGINS` | CORS allowed origins | No | localhost:3000,3001 |

## Troubleshooting

### Email Not Sending
1. Verify Gmail credentials are correct
2. Ensure 2FA is enabled and app password is generated
3. Check if Gmail account allows less secure apps (if needed)
4. Verify recipient email is correct

### CORS Issues
1. Check `ALLOWED_ORIGINS` includes your frontend domain
2. Verify frontend is sending correct headers
3. Check browser console for CORS errors

### cPanel Deployment Issues
1. Ensure Node.js is enabled and correct version is selected
2. Verify all files are uploaded correctly
3. Check cPanel error logs for specific issues
4. Ensure dependencies are installed via npm install

## Support

For issues related to this backend service, check the console logs for detailed error messages. Ensure all environment variables are properly configured and Gmail integration is set up correctly.

## License

ISC License
